/* =============================================================================
 *
 * MainTask.c
 *
 * This file contains the code for the main (menu) task (Main_Task) plus the global
 * declarations of the event groups for Main_Task and the demos.
 *
 * v1.0	YFS	Initial release. 
 *
 * =============================================================================
*/

/*
* Copyright (2014), Cypress Semiconductor Corporation. All Rights Reserved.
*
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express written permission of Cypress.
*
* Disclaimer: CYPRESS MAKES NO WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, WITH 
* REGARD TO THIS MATERIAL, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
* Cypress reserves the right to make changes without further notice to the 
* materials described herein. Cypress does not assume any liability arising out 
* of the application or use of any product or circuit described herein. Cypress 
* does not authorize its products for use as critical components in life-support 
* systems where a malfunction or failure may reasonably be expected to result in 
* significant injury to the user. The inclusion of Cypress' product in a life-
* support systems application implies that the manufacturer assumes all risk of 
* such use and in doing so indemnifies Cypress against all charges. 
*
* Use of this Software may be limited by and subject to the applicable Cypress
* software license agreement. 
*/

/* Include FreeRTOS APIs and defines */
#include <FreeRTOS.h>
#include <task.h>
#include <queue.h>
#include <event_groups.h>

/* Include PSoC system and component APIs and defines */
#include <project.h>

/* Include application function declarations and defines */
#include <utils.h>
#include <App_Defs.h>
#include <Task_Defs.h>

/* Declare the event groups for Main_Task and the demos */
EventGroupHandle_t Demo_Events;
EventGroupHandle_t Main_Task_Event;

/* Declare an array of demo names. Used to display the demo menu */
static char * const demoNames[NUM_DEMOS] =
{
	GLOWINGLED_DEMO_TASK,
	ADC_SAR_DEMO_TASK,
	ADC_DELSIG_DEMO_TASK,
	CAPSENSE_DEMO_TASK,
	DIGITAL_SCOPE_DEMO_TASK
};

/*
 * Function:		Main_Task
 *
 * This function runs as a FreeRTOS task with a low priority. It creates the
 * demo tasks, event groups, and the CapSense monitor task and queue. The main
 * loop displays the menu of demos and allows them to be browsed via CapSense
 * buttons (up/down) and started (SW2).
 *
 * Execution of this task is controlled by by an event group. Once a demo has
 * been started Main_Task (if it gets any execution time at all) pends on its
 * event bit so that it will not run while the demo is in a delay call (or is
 * not running for any other reason). The demo restarts Main_Task by posting to
 * its event bit.
 *
 *			Start task
 *			Create CapSense queue and the event groups
 *			Create all demo tasks (medium priority)
 *			Create CapSense monitor task (high priority)
 *			Clear LCD and print demo text
 *			Print demo instructions on LCD
 *		 -> if demo is starting, or restarting, refresh the LCD
 *		|	Get data from the CapSense queue (non-blocking)
 *		|	If a button was pressed change the displayed demo (up/down)
 *		|   Check if SW2 is pressed. If so...
 *		|   	Flush the CapSense queue
 *		|   	Set the demo task event flag (make it run)
 *		|   	Wait on own event (sleep)
 *      |   	Flush the CapSense queue
 *		|		Reset displayed demo to #1
 *		 -- Loop
 * 
 * Globals:		Uses Demo_Events, Main_Task_Event and CapSense_Monitor_Q
 *
 * Parameters:	pvParameters (not used)
 *
 * Return:		None (infinite loop)
 */
void Main_Task( void *pvParameters )
{
	CS_PACKET demo;		    /* Hold a message from CapSense Q */
       
    BaseType_t queueDataReceived;   /* Return from xQueueCreate */
    
    BaseType_t taskCreated;         /* Return from xTaskCreate */
	
	int demoIndex = 0;	    /* Currently selected demo (0 means demo #1) */

	int initScreen = 1;	    /* Flag to show the intro text on LCD */
	
	/* Start the LCD and print the welcome text */ 
	LCD_Start();
	print_welcome();
    	
	/*
	Create the OS resources used by the applications
	*/
	
	/* Create the CapSense_queue - must create it prior to tasks that use it */
    CapSense_Monitor_Q = xQueueCreate(
        CAPSENSE_MONITOR_Q_SIZE,
        sizeof( CS_PACKET ) );
    
    if( ! CapSense_Monitor_Q )
    {
        report_error( "Queue create" );
    }
    
    /* Create the event group to control which demo to run */
    Demo_Events = xEventGroupCreate();
    
    if( ! Demo_Events )
    {
        report_error( "Event create" );
    }
    
    /* Create the event group that stops this task from running while the
    demos are started but inactive */
    Main_Task_Event = xEventGroupCreate();
    
    if( ! Main_Task_Event )
    {
        report_error( "Event create" );
    }
    
    /*
    Create the application tasks and CapSense monitor
    */
    
	/* Create GlowingLED_Demo_Task */   
	taskCreated = xTaskCreate(
        GlowingLED_Demo_Task,
        GLOWINGLED_DEMO_TASK,
        DEMO_TASK_STACK_SIZE,
        (void *)demoIndex,          /* Pass the index as a task argument*/
        DEMO_TASK_PRIORITY,
        TASK_HANDLE_UNUSED );
    
    if( taskCreated != pdPASS )
    {
        report_error( "LED task" );
    }    
	
	/* Create ADC_SAR_Demo_Task */
    demoIndex++;                    /* Increment the task argument */
	taskCreated = xTaskCreate(
        ADC_SAR_Demo_Task,
        ADC_SAR_DEMO_TASK,
        DEMO_TASK_STACK_SIZE,
        (void *)demoIndex,          /* Pass the index as a task argument*/
        DEMO_TASK_PRIORITY,
        TASK_HANDLE_UNUSED );
    
    if( taskCreated != pdPASS )
    {
        report_error( "SAR task" );
    } 
    
	/* Create ADC_DelSig_Demo_Task */
    demoIndex++;                    /* Increment the task argument */
	taskCreated = xTaskCreate(
        ADC_DelSig_Demo_Task,
        ADC_DELSIG_DEMO_TASK,
        DEMO_TASK_STACK_SIZE,
        (void *)demoIndex,          /* Pass the index as a task argument*/
        DEMO_TASK_PRIORITY,
        TASK_HANDLE_UNUSED );  
    
    if( taskCreated != pdPASS )
    {
        report_error( "DelSig task" );
    }   
    
	/* Create CapSense_Demo_Task */
    demoIndex++;                    /* Increment the task argument */
	taskCreated = xTaskCreate(
        CapSense_Demo_Task,
        CAPSENSE_DEMO_TASK,
        DEMO_TASK_STACK_SIZE,
        (void *)demoIndex,          /* Pass the index as a task argument*/
        DEMO_TASK_PRIORITY,
        TASK_HANDLE_UNUSED );  
    
    if( taskCreated != pdPASS )
    {
        report_error( "CapSense task" );
    } 
    
	/* Create Digital_Scope_Demo_Task */
    demoIndex++;                    /* Increment the task argument */
	taskCreated = xTaskCreate(
        Digital_Scope_Demo_Task,
        DIGITAL_SCOPE_DEMO_TASK,
        DEMO_TASK_STACK_SIZE,
        (void *)demoIndex,          /* Pass the index as a task argument*/
        DEMO_TASK_PRIORITY,
        TASK_HANDLE_UNUSED );  
    
    if( taskCreated != pdPASS )
    {
        report_error( "Scope task" );
    }
	
	/* Create CapSense_Monitor_Task */
	taskCreated = xTaskCreate(
        CapSense_Monitor_Task,
        CAPSENSE_MONITOR_TASK,
        DEMO_TASK_STACK_SIZE,
        NO_TASK_ARG,                /* No argument needed for this task */
        DEMO_TASK_PRIORITY,
        TASK_HANDLE_UNUSED );  
    
    if( taskCreated != pdPASS )
    {
        report_error( "Monitor task" );
    }
	
	/* Reset demoIndex to the start of the array */
    demoIndex = 0;
    
	/* Flush any early scan data from the CapSense queue */
    xQueueReset( CapSense_Monitor_Q );
	
	/* By this point, the OS is running and all tasks have been created */
	
	/* Print out the instructions for using the demo */		
	scroll( 1, "Use CapSense buttons to scroll through demos. Use SW2 to start and SW3 to finish a demo." );
	
	for( ; ; )		/* Main loop (forever) */
	{
		/* Run this code the first time and when a demo completes */
		if( initScreen )
		{
			initScreen = 0;
			
			/* Print the welcome message */
			print_welcome();
			
			/* Print the demo title again */
			show_demo_title( demoIndex, demoNames[demoIndex] );
		}
		
		/* Collect the CapSense activity from the message queue */
        queueDataReceived = xQueueReceive(
            CapSense_Monitor_Q,
            &demo.result,           /* Put queue data here */
            DO_NOT_WAIT_ON_QUEUE );
        
        if( queueDataReceived == pdTRUE )        
		{
			/* Move the index based on the button press */
			if( demo.b0 )
				demoIndex--;		/* Move up */
			else if( demo.b1 )
				demoIndex++;		/* Move down */
			
			/* Make the menu circular */
			if( demoIndex < MIN_DEMO )
				demoIndex = MAX_DEMO;
			else if( demoIndex > MAX_DEMO )
				demoIndex = MIN_DEMO;	
				
			/* Clear the LCD and display the new demo name */
			show_demo_title( demoIndex, demoNames[demoIndex] );
		} /* if( queueDataReceived == pdTRUE ) */
		
		
		/* Check to see if SW2 was pressed - this signals a demo selection */
		if( ! P6_1_Read() )
		{
			/* Flush any CapSense data from queue - it is no longer relvant */
			xQueueReset( CapSense_Monitor_Q );
			
			/* Set the selected demo's event bit so it can run */
            xEventGroupSetBits( Demo_Events, 1 << demoIndex );
            
            /* Stall Main_Task while the demo runs (even if demo delays) */
            xEventGroupWaitBits(
                Main_Task_Event,
                MAIN_TASK_EVENT_NUM,    /* Event bit for Main_Task */
                pdTRUE,                 /* Clear the event on exit */
                pdFALSE,                /* Do not wait for all bits (only 1) */
                portMAX_DELAY );        /* Wait forever */
			
			/* Flush any CapSense data from queue - it is no longer relvant */
			xQueueReset( CapSense_Monitor_Q );
			
			initScreen = 1;	            /* Re-display the menu from the start */
            
		} /* if( ! P6_1_Read() ) */
		
	} /* for( ; ; ) */

} /* Main_Task() */
